/*
 * Common code for all drivers using VMCF
 */
#include "../h/param.h"
#include "../h/vmcf.h"
#include "../h/ioconf.h"

#define VMCPRI  (PZERO+1)

struct vmcparm vmcparm;         /* VMCF parameter list */
struct vmcmhdr vmcmhdr;         /* VMCF message header */
int vmc_count;                  /* # of current connections */
int vmc_mid = 1;                /* unique, non-zero message id */

/*
 * Things we need to know about a VMCF connection
 */
struct vmc_data {
	char    d_user[8];      /* EBCDIC userid */
	int     (*d_intr)();    /* interrupt rtn addr */
	int     d_arg;          /* arg for intr rtn */
	int     d_mid;          /* message ID of active req */
	int     d_eflg;         /* data transfer error # */
	int     d_flag;         /* flag */
	int     d_resid;        /* residual length of reply */
} vmc_data[NVMCF];

/*
 * Catch VMCF messages to and from a user
 * Equivalent to an open
 * Returns a descriptor used in further calls
 */
vmc_catch(user, intr, arg)
char *user;
int (*intr)();
int arg;
{
	int i, j;
	register struct vmc_data *dp;
	extern char atetab[128];

	for(i=0; i<NVMCF; i++) {
		dp = &vmc_data[i];
		if(dp->d_flag == 0) {
			dp->d_flag = 1;
			dp->d_intr = intr;
			dp->d_arg = arg;
			if(user != NULL) {
				for(j=0; j<8; j++) {
					/* translate, upcase, blank pad all at once */
					/* assumes only alphanumerics in userid */
					dp->d_user[j] = atetab[*user] | 0x40;
					if(*user) user++;
				}
			} else {
				dp->d_user[0] = 0;
			}
			if(vmc_count == 0 && vmc_auth() != 0) {
                                dp->d_flag = 0;
                                return(-1);
			}
			vmc_count++;
			return(i);
		}
	}
	return(-1);
}

/*
 * Drop a VMCF connection
 * (Free table slot)
 */
vmc_drop(d)
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;

	dp->d_flag = 0;
	/*
	 * Cancel any outstanding request
	 */
	if(dp->d_mid != 0) {
		p->p_func = P_CANC;
		p->p_mid = dp->d_mid;
		cpy8(p->p_user, dp->d_user);
		d_vmcf(p);
		dp->d_mid = 0;
	}
	if(vmc_count > 0 && --vmc_count == 0)
		vmc_uaut();
}

/*
 * Authorize for VMCF usage
 * Done on first vmc_catch
 */
static vmc_auth()
{
	register struct vmcparm *p = &vmcparm;
	int e;

	p->p_flg1 = P_PRTY+P_SMSG;
	p->p_func = P_AUTH;
	p->p_vada = (caddr_t)&vmcmhdr;
	p->p_lena = sizeof vmcmhdr;
	if(e = d_vmcf(p))
		printf("VMCF: authorization failed\n");
	return(e);
}

/*
 * Unauthorize for VMCF usage
 * Done on last vmc_drop
 */
static vmc_uaut()
{
	register struct vmcparm *p = &vmcparm;

	p->p_func = P_UAUT;
	d_vmcf(p);
}

/*
 * All VMCF interrupts come here and are passed on appropriately
 * 'vmcmhdr' contains information relevant to the interrupt
 * Reject any unwanted interrupts
 */
vmcfint()
{
	register struct vmcmhdr *h = &vmcmhdr;
	register struct vmc_data *dp;
	register struct vmcparm *p = &vmcparm;

	for(dp = &vmc_data[0]; dp < &vmc_data[NVMCF]; dp++)
	if(cmp8(h->m_user, dp->d_user)) {
		if((h->m_stat & M_RESP) && h->m_mid == dp->d_mid) {
			/* interrupt after send */
			dp->d_mid = 0;
			dp->d_eflg = h->m_eflg;
			dp->d_resid = h->m_lenb;
			wakeup((caddr_t)dp);
			return;
		}
		if((*dp->d_intr)(dp->d_arg, h))
			return;
	}
	if(h->m_func == P_SEND || h->m_func == P_SENR) {
		p->p_func = P_RJCT;
		p->p_mid = h->m_mid;
		cpy8(p->p_user, h->m_user);
		d_vmcf(p);
	}
}

/*
 * Issue VMCF SEND request
 */
vmc_send(d, mid, buf, len)
int d, mid, len;
caddr_t buf;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;
	register int e;

	while(dp->d_mid != 0)
		sleep((caddr_t)dp, VMCPRI);
	if(mid != 0)
		dp->d_mid = mid;
	else
		dp->d_mid = vmc_mid++;
	dp->d_eflg = 0;
	p->p_func = P_SEND;
	p->p_mid = dp->d_mid;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	p->p_vada = buf;
	p->p_lena = len;
	e = d_vmcf(p);
	if(e)
		return(e);
	while(dp->d_mid != 0)
		sleep((caddr_t)dp, VMCPRI);
	return(dp->d_eflg);
}

/*
 * Issue VMCF SEND/RECV request
 */
vmc_senr(d, mid, xbuf, xlen, rbuf, rlen, resid)
int d, mid, xlen, rlen, *resid;
caddr_t xbuf, rbuf;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;
	register int e;

	while(dp->d_mid != 0)
		sleep((caddr_t)dp, VMCPRI);
	if(mid != 0)
		dp->d_mid = mid;
	else
		dp->d_mid = vmc_mid++;
	dp->d_eflg = 0;
	p->p_func = P_SENR;
	p->p_mid = dp->d_mid;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	p->p_vada = xbuf;
	p->p_lena = xlen;
	p->p_vadb = rbuf;
	p->p_lenb = rlen;
	e = d_vmcf(p);
	if(e)
		return(e);
	while(dp->d_mid != 0)
		sleep((caddr_t)dp, VMCPRI);
	*resid = dp->d_resid;
	return(dp->d_eflg);
}

/*
 * Issue VMCF SENDX request
 */
vmc_senx(d, mid, buf, len)
int d, mid, len;
caddr_t buf;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;
	register int e;

	while(dp->d_mid != 0)
		sleep((caddr_t)dp, VMCPRI);
	if(mid != 0)
		dp->d_mid = mid;
	else
		dp->d_mid = vmc_mid++;
	dp->d_eflg = 0;
	p->p_func = P_SENX;
	p->p_mid = dp->d_mid;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	p->p_vada = buf;
	p->p_lena = len;
	e = d_vmcf(p);
	if(e)
		return(e);
	while(dp->d_mid != 0)
		sleep((caddr_t)dp, VMCPRI);
	return(dp->d_eflg);
}

/*
 * Issue VMCF RECEIVE request
 */
vmc_recv(d, mid, buf, len)
int d, mid, len;
caddr_t buf;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;

	p->p_func = P_RECV;
	p->p_mid = mid;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	p->p_vada = buf;
	p->p_lena = len;
	return(d_vmcf(p));
}

/*
 * Issue VMCF REPLY request
 */
vmc_repl(d, mid, buf, len)
int d, mid, len;
caddr_t buf;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;

	p->p_func = P_REPL;
	p->p_mid = mid;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	p->p_vada = buf;
	p->p_lena = len;
	return(d_vmcf(p));
}

/*
 * Issue VMCF REJECT request
 */
vmc_rjct(d, mid)
int d, mid;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;

	p->p_func = P_RJCT;
	p->p_mid = mid;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	return(d_vmcf(p));
}

/*
 * Issue VMCF CANCEL request
 */
vmc_canc(d, mid)
int d, mid;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;

	p->p_func = P_CANC;
	p->p_mid = mid;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	return(d_vmcf(p));
}

/*
 * Issue VMCF IDENTIFY request
 */
vmc_iden(d)
int d;
{
	register struct vmc_data *dp = &vmc_data[d];
	register struct vmcparm *p = &vmcparm;

	p->p_func = P_IDEN;
	p->p_flg1 = 0;
	cpy8(p->p_user, dp->d_user);
	return(d_vmcf(p));
}

cmp8(x, y)
char *x, *y;
{
	register i;

	for(i=0; i<8; i++)
		if(*x++ != *y++)
			return(0);
	return(1);
}

cpy8(x, y)
char *x, *y;
{
	register i;

	for(i=0; i<8; i++)
		*x++ = *y++;
}
