/*
 * LDSF - Logical Device Support Facility
 * CP Diagnose code x'7C'
 * Provides virtual CP terminals, sort of
 * Only 3277-2 type terminals are supported here
 */

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/conf.h"
#include "../h/buf.h"
#include "../h/tty.h"
#include "../h/ioconf.h"

struct ldev {
	short   ld_addr;        /* logical device address */
	char    ld_code;        /* interrupt code */
	caddr_t ld_ibuf;        /* input buffer */
	caddr_t ld_obuf;        /* output buffer */
	int     ld_olen;        /* output length and flag bit */
	int     ld_flag;        /* flags */
	int     ld_pksig;       /* signal for poking */
	struct proc *ld_pkproc; /* process for poking */
} ldev[NLDEV];

/* ld_flag bits */
#define OPEN    1       /* device in use */
#define CODE    2       /* interrupt to be serviced */
#define OFULL   4       /* output buffer is full */

#define T3277M2 0x00024004      /* model, class, and type of 3277-2 */
#define READBUF 0x02            /* ccw op for read buffer */
#define READMOD 0x06            /* ccw op for read modified */

/*
 * Open logical device
 * Does INITIATE subcommand and grabs buffers
 */
ldsopen(dev)
{
	register struct ldev *ld;
	int rc;

	if(minor(dev) >= NLDEV) {
		u.u_error = ENXIO;
		return;
	}
	ld = &ldev[minor(dev)];
	if(ld->ld_flag & OPEN) {
		u.u_error = EBUSY;
		return;
	}
	rc = d_ldsfi(T3277M2, &ld->ld_addr);    /* sets ld_addr */
	if(rc != 0) {
		u.u_error = EIO;
		return;
	}
	ld->ld_flag = OPEN;
	ld->ld_pksig = 0;
	ld->ld_ibuf = getpage();
	ld->ld_obuf = getpage();
}

/*
 * Close logical device
 * Issue TERMINATE subcommand and free buffers
 */
ldsclose(dev)
{
	register struct ldev *ld = &ldev[minor(dev)];

	ld->ld_flag = 0;
	d_ldsft(ld->ld_addr);
	freepag(ld->ld_ibuf);
	freepag(ld->ld_obuf);
}

/*
 * Read from logical device
 * (CP writes to virtual terminal)
 */
ldsread(dev)
{
	register struct ldev *ld = &ldev[minor(dev)];
	int rc, n;

tryagain:
	while((ld->ld_flag & CODE) == 0)
		sleep((caddr_t)&ld->ld_flag, TTIPRI);
	ld->ld_flag &= ~CODE;
	switch(ld->ld_code) {

	case 1: /* terminal disconnected, return EOF */
		break;

	case 2: /* issue ACCEPT subcommand */
		rc = d_ldsfa(ld->ld_addr, ld->ld_ibuf, PSIZE, &n);
		switch(rc) {
		case 1: /* OK, but more to come */
			ld->ld_flag |= CODE;
			ld->ld_code = 2;
			/* fall-through */
		case 0: /* OK */
			iomove(ld->ld_ibuf, n, B_READ);
			break;
		case (1<<8)+2:  /* no data yet */
			goto tryagain;
		case (3<<8)+1:  /* dead terminal, return EOF */
			break;
		default:
			printf("ldsf accept rc=%x\n", rc);
			u.u_error = EIO;
			break;
		}
		break;

	case 4:
		passc(READBUF);
		break;

	case 5:
		passc(READMOD);
		break;
	}
}

/*
 * Write to logical device
 * (CP reads from virtual terminal)
 */
ldswrite(dev)
{
	register struct ldev *ld = &ldev[minor(dev)];
	int rc;

	while(ld->ld_flag & OFULL)
		sleep((caddr_t)&ld->ld_flag, TTOPRI);
	ld->ld_flag |= OFULL;
	ld->ld_olen = min(PSIZE, u.u_count) - 1;
	iomove(ld->ld_obuf, ld->ld_olen+1, B_WRITE);
	if(ld->ld_obuf[0] == READBUF)
		ld->ld_olen |= 0x80000000;
	rc = d_ldsfp(ld->ld_addr, ld->ld_obuf+1, ld->ld_olen);
	switch(rc) {
	case (2<<8)+1:  /* pending data must be accepted */
		ld->ld_code = 2;
		ld->ld_flag |= CODE;
		break;
	case (2<<8)+2:  /* previous PRESENT active */
		/* leave OFULL on and return */
		return;
	case (2<<8)+3:  /* CP wants READBUF */
	case (2<<8)+4:  /* CP wants READMOD */
		u.u_error = EINVAL;
		break;
	case (3<<8)+1:  /* device being terminated */
	case (3<<8)+2:  /* bad device address */
		u.u_error = EIO;
		break;
	}
	ld->ld_flag &= ~OFULL;
	wakeup((caddr_t)&ld->ld_flag);
}

/*
 * ioctl for ldevs
 * Polling and poking
 */
/* ARGSUSED */
ldsioctl(dev, cmd, argp, flag)
dev_t dev;
caddr_t argp;
{
	register struct ldev *ld = &ldev[minor(dev)];

	switch(cmd) {

	case TIOPOLL:
		suword(argp, (ld->ld_flag & CODE) ? 1 : 0);
		break;

	case TIOPOKE:
		ld->ld_pksig = (int)argp;
		ld->ld_pkproc = u.u_procp;
		break;

	case TIONPOKE:
		ld->ld_pksig = 0;
		break;

	default:
		u.u_error = ENOTTY;
		break;
	}
}

/*
 * Low core communication area for external interrupt x'2401'
 */
struct ldsfcrud {
	short   ldaddr;
	char    ldflag;
	char    ldcode;
};

#define LDSFPTR ((struct ldsfcrud *)128)

/*
 * Interrupt handler for logical devices
 */
ldsfint()
{
	register struct ldev *ld;

	for(ld = &ldev[0]; ld < &ldev[NLDEV]; ld++)
	if(ld->ld_addr == LDSFPTR->ldaddr && (ld->ld_flag & OPEN)) {
		if(LDSFPTR->ldcode == 3) {      /* retry PRESENT */
			if(ld->ld_flag & OFULL) {
				d_ldsfp(ld->ld_addr, ld->ld_obuf+1, ld->ld_olen);
				ld->ld_flag &= ~OFULL;
				wakeup((caddr_t)&ld->ld_flag);
			}
		} else {
			ld->ld_code = LDSFPTR->ldcode;
			ld->ld_flag |= CODE;
			wakeup((caddr_t)&ld->ld_flag);
			if(ld->ld_pksig > 0) {
                                psignal(ld->ld_pkproc, ld->ld_pksig);
				ld->ld_pksig = 0;
			}
		}
	}
}
