/*
 *      offline printer filter
 *      removes VT   /ESC VT for positive/negative full line feeds,
 *              ESC D/ESC U  for positive/negative half line feeds,
 *              ESC G/ESC 4  for graphics mode on/off,
 *                           grahics mode partial hor/ver motion,
 *              SO   /SI     for boldface on/off.
 */

#include <stdio.h>

			 /********************************************/
#define SPACE     040    /* SP                                       */
#define BACK_SPC  010    /* BS                                       */
#define TAB       011    /* HT                                       */
#define CARG_RET  015    /* CR                                       */
#define NEWLINE   012    /* LF becomes CR LF                         */
#define LINEFEED  013    /* VT for LF                                */
#define BOLD      016    /* SO for begin bold font                   */
#define ROMAN     017    /* SI for begin roman font                  */
#define SHIFT_UP  002    /* to simulate subscripts/superscripts      */
#define SHIFT_DN  003    /* to simulate subscripts/superscripts      */
#define ESCAPE    033    /* ESC for simulating rev half line feeds   */
#define NEG_LF    013    /* escaped VT for reverse line feed         */
#define POS_HLF   'U'    /* escaped U  for forward half line         */
#define NEG_HLF   'D'    /* escaped D  for reverse half line         */
#define ON        'G'    /* escaped 'G' for graphics mode on         */
#define OFF       '4'    /* escaped '4' for graphics mode off        */
#define V_RES       8    /* vertical resolution per row              */
#define H_RES      10    /* horizontal resolution per column         */
#define SIZE     4096    /* size of array string                     */
			 /********************************************/
char **page;             /* page[row][0:wdth[row]-1]                 */
int row, col;            /* position to place next character         */
int width  = 132;        /* page width,  default 132 spaces/line     */
int length =  66;        /* page length, default  66 lines/page      */
int offset =   0;        /* page offset, default   0 spaces          */
int *wdth;               /* array of widths of each row              */
int written;             /* number of graphic chars on current page  */
char string[SIZE];       /* input string being read                  */
int p;                   /* length of string (position of next char) */
int start;               /* column in which string starts            */
int graphics;            /* graphics mode is ON or OFF               */
int shift;               /* shift up for superscripts                */
int p_row;               /* 0:7, partial distance above current row  */
int p_col;               /* 0:9, partial distance right of col       */
int c;                   /* current input character                  */
char font;               /* current font, BOLD or ROMAN              */
char *malloc();          /* standard dynamic memory allocation       */
char *realloc();         /* standard dynamic memory re-allocation    */
char *name;              /* command name                             */
			 /********************************************/

main(argc, argv)
int argc;
char *argv[];
{
	FILE *fp;

	argc--;
	name = *argv++;

	while (argc && argv[0][0] == '-') {
		switch (argv[0][1]) {
		case 'w':
			width  = atoi(*argv+2);
			break;
		case 'l':
			length = atoi(*argv+2);
			break;
		case 'o':
			offset = atoi(*argv+2);
			break;
		default:
			fprintf(stderr, "%s: invalid %s option\n",
				name, *argv);
			exit(1);
			break;
		}
		argc--;
		argv++;
	}

	page = (char **) malloc(length*sizeof(char *));
	wdth = (int   *) malloc(length*sizeof(int   ));
	if (page==NULL || wdth==NULL) {
		fprintf(stderr, "%s: out of memory\n", name);
		exit(1);
	}

	page--;
	wdth--;
	for (row=1; row<=length; row++) {
		page[row] = malloc(0);
		if (page[row]==NULL) {
			fprintf(stderr, "%s: out of memory\n", name);
			exit(1);
		}
		wdth[row] = 0;
	}
	row     = 1;
	col     = 1+offset;
	start   = col;
	graphics= OFF;
	shift   = 0;
	p_row   = 0;
	p_col   = 0;
	p       = 0;
	written = 0;
	font    = ROMAN;

	if (argc) {
		while(argc) {
			fp = fopen(*argv, "r");
			if (fp == NULL) {
				fprintf(stderr, "%s: cannot open %s\n",
					name, *argv);
				exit(1);
			}
			oprf(fp);
			fclose(fp);
			argv++;
			argc--;
		}
	} else {
		if (isatty(fileno(stdin))) {
			fprintf(stderr, "%s: no files specified\n", name);
			exit(1);
		} else {
			oprf(stdin);
		}
	}

	enter();
	if (written)
		output();
}

oprf(fp)
FILE *fp;
{
	int tabspace;

	for (;;) {
		switch(c = getc(fp)) {
		case SPACE:
			if (graphics == ON) {
				p_col++;
				if (p_col >= H_RES) {
					p_col -= H_RES;
					if (1<=col && col<=width)
						string[p++] = c;
					col++;
				}
			} else {
				if (1<=col && col<=width)
					string[p++] = c;
				col++;
			}
			break;
		case BACK_SPC:
			if (graphics == ON) {
				p_col--;
				if (p_col < 0) {
					p_col += H_RES;
					if (1<=col && col<=width)
						string[p++] = c;
					col--;
				}
			} else {
				if (1<=col && col<=width)
					string[p++] = c;
				col--;
			}
			break;
		case TAB:
			tabspace = -((col-1) % 8);
			if (tabspace <= 0)
				tabspace += 8;
			for (c=1; c<=tabspace; c++) {
				if (1<=col && col<=width)
					string[p++] = SPACE;
				col++;
			}
			p_col = 0;
			break;
		case CARG_RET:
			col = 1+offset;
			p_col = 0;
			break;
		case NEWLINE:
			col = 1+offset;
			p_col = 0;
			if (graphics == ON)
				linefeed(1);
			else
				linefeed(V_RES);
			break;
		case LINEFEED:
			if (graphics == ON)
				linefeed(1);
			else
				linefeed(V_RES);
			break;
		case BOLD:
		case ROMAN:
			font = c;
			break;
		case ESCAPE:
			switch(c = getc(fp)) {
			case NEG_LF:
				if (graphics == ON)
					linefeed(-1);
				else
					linefeed(-V_RES);
				break;
			case POS_HLF:
				linefeed(V_RES/2);
				break;
			case NEG_HLF:
				linefeed(-V_RES/2);
				break;
			case ON:
			case OFF:
				graphics = c;
				break;
			}
			break;
		case EOF:
			return;
			break;
		case SHIFT_UP:
			shift++;
			break;
		case SHIFT_DN:
			shift--;
			break;
		default:
			if (c > SPACE) {
				if (1 <= col && col <= width) {
					if (shift>0 && (c=='+'
					 || c=='(' || c==')'
					 || (c>='0' && c<='9')))
						c |= 0200;
					if (font == BOLD) {
						string[p++] = c;
						string[p++] = BACK_SPC;
						string[p++] = c;
						string[p++] = BACK_SPC;
					}
					string[p++] = c;
				}
				if (graphics == OFF) {
					col++;
				}
				written++;
			}
			break;
		}
	}
}

linefeed(amount)
int amount;
{
	p_row -= amount;
	if (p_row < 0) {
		p_row += V_RES;
		enter();
		row++;
		if (row > length) {
			output();
			row     = 1;
			written = 0;
		}
	} else if (p_row >= V_RES) {
		p_row -= V_RES;
		enter();
		row--;
	}
}

enter()
{
	int new_wid;
	char *new_row;
	int i, j;
	if (p == 0 || row <= 0)
		return;
	new_wid = wdth[row] + start + p;
	if (offset < 0) {
		new_wid -= offset;
	}
	new_row = malloc(new_wid+16);
	if (new_row==NULL) {
		fprintf(stderr, "%s: out of memory\n", name);
		exit(1);
	}

	i = 0;

	for (j=0; j<wdth[row]; j++)
		new_row[i++] = page[row][j];

	if (wdth[row] == 0)
		new_wid--;
	else
		new_row[i++] = CARG_RET;

	for (j=1; j<start; j++)
		new_row[i++] = SPACE;

	for (j=0; j<p; j++)
		new_row[i++] = string[j];

	free(page[row]);
	wdth[row] = new_wid;
	page[row] = new_row;
	p     = 0;
	start = col;
}

output()
{
	int row, col;
	for (row=1; row<=length; row++) {
		for (col=0; col<wdth[row]; col++) {
			putchar(page[row][col]);
		}
		putchar('\n');
		page[row] = realloc(page[row], 0);
		wdth[row] = 0;
	}
}
